import type { Instrumenter } from './instrumenter';
import type { Primitive } from './misc';
import type { Transaction } from './transaction';
type SpanOriginType = 'manual' | 'auto';
type SpanOriginCategory = string;
type SpanOriginIntegrationName = string;
type SpanOriginIntegrationPart = string;
export type SpanOrigin = SpanOriginType | `${SpanOriginType}.${SpanOriginCategory}` | `${SpanOriginType}.${SpanOriginCategory}.${SpanOriginIntegrationName}` | `${SpanOriginType}.${SpanOriginCategory}.${SpanOriginIntegrationName}.${SpanOriginIntegrationPart}`;
/** Interface holding all properties that can be set on a Span on creation. */
export interface SpanContext {
    /**
     * Description of the Span.
     */
    description?: string;
    /**
     * Human-readable identifier for the span. Alias for span.description.
     */
    name?: string;
    /**
     * Operation of the Span.
     */
    op?: string;
    /**
     * Completion status of the Span.
     * See: {@sentry/tracing SpanStatus} for possible values
     */
    status?: string;
    /**
     * Parent Span ID
     */
    parentSpanId?: string;
    /**
     * Was this span chosen to be sent as part of the sample?
     */
    sampled?: boolean;
    /**
     * Span ID
     */
    spanId?: string;
    /**
     * Trace ID
     */
    traceId?: string;
    /**
     * Tags of the Span.
     */
    tags?: {
        [key: string]: Primitive;
    };
    /**
     * Data of the Span.
     */
    data?: {
        [key: string]: any;
    };
    /**
     * Timestamp in seconds (epoch time) indicating when the span started.
     */
    startTimestamp?: number;
    /**
     * Timestamp in seconds (epoch time) indicating when the span ended.
     */
    endTimestamp?: number;
    /**
     * The instrumenter that created this span.
     */
    instrumenter?: Instrumenter;
    /**
     * The origin of the span, giving context about what created the span.
     */
    origin?: SpanOrigin;
}
/** Span holding trace_id, span_id */
export interface Span extends SpanContext {
    /**
     * Human-readable identifier for the span. Identical to span.description.
     */
    name: string;
    /**
     * @inheritDoc
     */
    spanId: string;
    /**
     * @inheritDoc
     */
    traceId: string;
    /**
     * @inheritDoc
     */
    startTimestamp: number;
    /**
     * @inheritDoc
     */
    tags: {
        [key: string]: Primitive;
    };
    /**
     * @inheritDoc
     */
    data: {
        [key: string]: any;
    };
    /**
     * The transaction containing this span
     */
    transaction?: Transaction;
    /**
     * The instrumenter that created this span.
     */
    instrumenter: Instrumenter;
    /**
     * Sets the finish timestamp on the current span.
     * @param endTimestamp Takes an endTimestamp if the end should not be the time when you call this function.
     */
    finish(endTimestamp?: number): void;
    /**
     * Sets the tag attribute on the current span.
     *
     * Can also be used to unset a tag, by passing `undefined`.
     *
     * @param key Tag key
     * @param value Tag value
     */
    setTag(key: string, value: Primitive): this;
    /**
     * Sets the data attribute on the current span
     * @param key Data key
     * @param value Data value
     */
    setData(key: string, value: any): this;
    /**
     * Sets the status attribute on the current span
     * See: {@sentry/tracing SpanStatus} for possible values
     * @param status http code used to set the status
     */
    setStatus(status: string): this;
    /**
     * Sets the status attribute on the current span based on the http code
     * @param httpStatus http code used to set the status
     */
    setHttpStatus(httpStatus: number): this;
    /**
     * Set the name of the span.
     */
    setName(name: string): void;
    /**
     * Creates a new `Span` while setting the current `Span.id` as `parentSpanId`.
     * Also the `sampled` decision will be inherited.
     */
    startChild(spanContext?: Pick<SpanContext, Exclude<keyof SpanContext, 'sampled' | 'traceId' | 'parentSpanId'>>): Span;
    /**
     * Determines whether span was successful (HTTP200)
     */
    isSuccess(): boolean;
    /** Return a traceparent compatible header string */
    toTraceparent(): string;
    /** Returns the current span properties as a `SpanContext` */
    toContext(): SpanContext;
    /** Updates the current span with a new `SpanContext` */
    updateWithContext(spanContext: SpanContext): this;
    /** Convert the object to JSON for w. spans array info only */
    getTraceContext(): {
        data?: {
            [key: string]: any;
        };
        description?: string;
        op?: string;
        parent_span_id?: string;
        span_id: string;
        status?: string;
        tags?: {
            [key: string]: Primitive;
        };
        trace_id: string;
    };
    /** Convert the object to JSON */
    toJSON(): {
        data?: {
            [key: string]: any;
        };
        description?: string;
        op?: string;
        parent_span_id?: string;
        span_id: string;
        start_timestamp: number;
        status?: string;
        tags?: {
            [key: string]: Primitive;
        };
        timestamp?: number;
        trace_id: string;
    };
}
export {};
//# sourceMappingURL=span.d.ts.map